<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;
use Cake\I18n\FrozenTime;

/**
 * Loanmasters Model
 *
 * @method \App\Model\Entity\Loanmaster newEmptyEntity()
 * @method \App\Model\Entity\Loanmaster newEntity(array $data, array $options = [])
 * @method \App\Model\Entity\Loanmaster[] newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\Loanmaster get($primaryKey, $options = [])
 * @method \App\Model\Entity\Loanmaster findOrCreate($search, ?callable $callback = null, $options = [])
 * @method \App\Model\Entity\Loanmaster patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method \App\Model\Entity\Loanmaster[] patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\Loanmaster|false save(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Loanmaster saveOrFail(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Loanmaster[]|\Cake\Datasource\ResultSetInterface|false saveMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Loanmaster[]|\Cake\Datasource\ResultSetInterface saveManyOrFail(iterable $entities, $options = [])
 * @method \App\Model\Entity\Loanmaster[]|\Cake\Datasource\ResultSetInterface|false deleteMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Loanmaster[]|\Cake\Datasource\ResultSetInterface deleteManyOrFail(iterable $entities, $options = [])
 */
class LoanmastersTable extends Table
{
    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('loanmasters');
        $this->setDisplayField('id');
        $this->setPrimaryKey('id');
		
		$this->belongsTo('Members', [
            'foreignKey' => 'member_id',
            'joinType' => 'INNER',
        ]);
		
		$this->hasMany('Loandetails', [
            'foreignKey' => 'LoanMasterId',
        ]);
		
		$this->hasMany('Transactionmasters', [
            'foreignKey' => 'LoanMasterId',
        ]);
		
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->allowEmptyString('MemberId');

      /*   $validator
            ->scalar('LoanType')
            ->maxLength('LoanType', 10)
            ->allowEmptyString('LoanType');

        $validator
            ->decimal('ApprovedLoanAmount')
            ->allowEmptyString('ApprovedLoanAmount');

        $validator
            ->allowEmptyString('MonthsToPay');

        $validator
            ->decimal('ServiceFee')
            ->allowEmptyString('ServiceFee');

        $validator
            ->decimal('PercentInterest')
            ->allowEmptyString('PercentInterest');

        $validator
            ->decimal('Penalty')
            ->allowEmptyString('Penalty');

        $validator
            ->decimal('NetAmountOfLoan')
            ->allowEmptyString('NetAmountOfLoan');

        $validator
            ->decimal('Insurance')
            ->allowEmptyString('Insurance');

        $validator
            ->decimal('PercentCapitalBuildUp')
            ->allowEmptyString('PercentCapitalBuildUp');

        $validator
            ->decimal('PrePayments')
            ->allowEmptyString('PrePayments');

        $validator
            ->decimal('OtherDeductions')
            ->allowEmptyString('OtherDeductions');

        $validator
            ->decimal('MonthlyAmortization')
            ->allowEmptyString('MonthlyAmortization');

        $validator
            ->scalar('AmortizationStart')
            ->maxLength('AmortizationStart', 19)
            ->allowEmptyString('AmortizationStart');

        $validator
            ->scalar('AmortizationEnd')
            ->maxLength('AmortizationEnd', 19)
            ->allowEmptyString('AmortizationEnd');

        $validator
            ->decimal('OtherPayments')
            ->allowEmptyString('OtherPayments');

        $validator
            ->decimal('LessRebates')
            ->allowEmptyString('LessRebates');

        $validator
            ->scalar('ApprovedDate')
            ->maxLength('ApprovedDate', 19)
            ->allowEmptyString('ApprovedDate');

        $validator
            ->scalar('Status')
            ->maxLength('Status', 9)
            ->allowEmptyString('Status');

        $validator
            ->scalar('EntryDate')
            ->maxLength('EntryDate', 19)
            ->allowEmptyString('EntryDate');

        $validator
            ->scalar('UpdateDateTime')
            ->maxLength('UpdateDateTime', 19)
            ->allowEmptyString('UpdateDateTime');

        $validator
            ->decimal('OtherDeductionsPrePayment')
            ->allowEmptyString('OtherDeductionsPrePayment');

        $validator
            ->decimal('OtherDeductionsCanteen')
            ->allowEmptyString('OtherDeductionsCanteen');

        $validator
            ->scalar('Remarks')
            ->allowEmptyString('Remarks'); */

        return $validator;
    }
	
	
	public function getTotalApprovedLoanAmountForCurrentYear(): float
	{
        $currentYear = (new FrozenTime())->year;

        // Create a query to sum ApprovedLoanAmount where ApprovedDate is within the current year
        $total = $this->find()
            ->select(['total' => $this->find()->func()->sum('ApprovedLoanAmount')])
            ->where(function ($exp, $query) use ($currentYear) {
                return $exp->between('ApprovedDate', 
                    $currentYear . '-01-01', 
                    $currentYear . '-12-31',
                    'date'
                );
            })
            ->first()
            ->total;
	
		if(empty($total)){
			$total = 0;
		}
        return $total;
    }
	
	public function getLoanTotalsForCurrentYear(): array
    {
        $currentYear = (new FrozenTime())->year;

        // Query to get the total NetAmountOfLoan for the current year
        $totalNetAmount = $this->find()
            ->select(['total' => $this->find()->func()->sum('NetAmountOfLoan')])
            ->where(function ($exp, $query) use ($currentYear) {
                return $exp->between('ApprovedDate', 
                    $currentYear . '-01-01', 
                    $currentYear . '-12-31',
                    'date'
                );
            })
            ->first()
            ->total;

        // Query to get the total number of loans with a status of 'COMPLETED' for the current year
        $totalCompleted = $this->find()
            ->where([
                'Status' => 'COMPLETED',
                function ($exp, $query) use ($currentYear) {
                    return $exp->between('ApprovedDate', 
                        $currentYear . '-01-01', 
                        $currentYear . '-12-31',
                        'date'
                    );
                }
            ])
            ->count();

        return [
            'totalNetAmount' => $totalNetAmount,
            'totalCompleted' => $totalCompleted
        ];
    }
	
	
	 public function countIdsByMonth(): array
    {
        $currentYear = (new FrozenTime())->year;
        $counts = [];

        for ($month = 1; $month <= 12; $month++) {
            $start = new FrozenTime(sprintf('%d-%02d-01', $currentYear, $month));
            $end = $start->endOfMonth();

            $count = $this->find()
                ->where([
                    'ApprovedDate >=' => $start,
                    'ApprovedDate <=' => $end
                ])
                ->count();

            $counts[$start->format('F')] = $count;
        }

        return $counts;
    }
	
	public function computeTotalNetAmountOfLoan(): float
    {
        $totalNetAmount = $this->find()
            ->select(['sum' => $this->find()->func()->sum('NetAmountOfLoan')])
            ->first()
            ->sum;

        return $totalNetAmount;
    }
	
	 public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->isUnique(
			['member_id', 'LoanType', 'Status'],
			'Member has an active loan. Use reloan instead'
		), 
			['errorField' => 'contact_no']
		);
        //$rules->add($rules->isUnique(['email']), ['errorField' => 'email']);

        return $rules;
    }
	
	public function getActiveLoanByMemberId(int $memberId)
    {
        $query = $this->find()
            ->where([
                'member_id' => $memberId,
                //'Status' => 'ACTIVE'
            ])
            ->limit(1);

        return $query->first();
    }
	
	
	public function calculateInterest($amount, $months)
    {
        $annualInterestRate = 0.08; // 8% annual interest rate
        $years = $months / 12;
        $interest = $amount * $annualInterestRate * $years;

        return $interest;
    }
	
}
